#!/bin/bash

set -euo pipefail

pull_database() {
  local db_name="$1"
  export PGHOST="$2"
  export PGPORT="$3"
  export PGUSER="$4"
  export PGPASSWORD="$5"
  echo "Downloading '$db_name'..."
  pg_dump -Fc --clean --if-exists -h $PGHOST -p $PGPORT -U $PGUSER $db_name > /dump/"${db_name}".dump
}

push_database() {
  local file="$1"
  local db_name="${file%.dump}"
  echo "Uploading '$db_name'..."
  pg_restore -Fc --clean --if-exists -h $PGHOST -U $POSTGRES_USER -p $PGPORT -d $db_name < /dump/"${file}"
}

main() {
  if [ -d /dump ]; then rm -rf /dump; fi
  mkdir /dump
  pull_database $AUTH_DB_NAME authdb 5432 $AUTH_DB_USER $AUTH_DB_PASSWORD
  pull_database $AVATARS_DB_NAME avatars-db 5432 $AVATARS_DB_USER $AVATARS_PASSWORD
  pull_database $LICENSE_DB_NAME licensedb 5432 $LICENSE_DB_USER $LICENSE_DB_PASSWORD

  check="$(ls /dump | wc -l)"
  if [ "$check" != "3" ]; then
    echo "Import FAILED! Check database availability and postgres environment variables!"
    ls -l /dump
    exit 1
  fi

  export PGHOST="localhost"
  export PGPORT="5432"
  export PGUSER="${POSTGRES_USER}"
  export PGPASSWORD="${POSTGRES_PASSWORD}"
  echo "Initializing new databases..."
  psql -h $PGHOST -U $POSTGRES_USER -p $PGPORT -d $POSTGRES_DB < /init.sql
  files=$(ls /dump)
  for file in $files; do
    push_database $file
  done

  rm -rf /dump
  
  echo "Script finished!"
}

main "$@"
