#!/bin/bash
# pre_k8s_backup.sh 

set -euo pipefail

# Sources the common-functions.sh script from the directory path of the current script
source "$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"/common-functions.sh

backup_postgres() {
  echo "Backing up databases..."
  # Create backup folder
  backup_postgres_dir="${backup_dir}/postgres"
  mkdir -p "$backup_postgres_dir"
  # Start the Postgres container
  start_container "$postgres_container"
  sleep 10
  # Loop through databases and perform backups
  for db in "${db_names[@]}"; do
    docker_pgdump "$db"
  done
  # Stop the containers
  docker compose -f "${COMPOSE_FILE_PATH}" -p "${COMPOSE_PROJECT_PREFIX}" stop
}

# Function to perform the backup using pg_dump
# reconfigure for a single container
docker_pgdump() {
  POSTGRES_DB="$1"
  # Get PostgreSQL credentials from the container
  POSTGRES_PASSWORD=$(docker exec -i "$postgres_container" /bin/sh -c "printenv POSTGRES_PASSWORD")
  POSTGRES_USER=$(docker exec -i "$postgres_container" /bin/sh -c "printenv POSTGRES_USER")

  local backup_file="${backup_postgres_dir}/${POSTGRES_DB}_backup.sql"
  # Perform the backup using pg_dump with the retrieved credentials
  docker exec -i "$postgres_container" /bin/sh -c "PGPASSWORD='$POSTGRES_PASSWORD' pg_dump -Fc --clean --if-exists -h localhost -U '$POSTGRES_USER' '$POSTGRES_DB'" > "${backup_file}"
  if [ $? -eq 0 ]; then
    echo "'$POSTGRES_DB' backup saved."
  else
    echo "Backup failed for $POSTGRES_DB."
  fi
}

backup_minio() { 
  echo "Backing up buckets..."
  # Start the containers
  start_container "$minio_container" 
  docker_mc_export
  stop_container "$minio_container" 
  echo "Buckets backup finished."
}

# Function to launch the minio migration job
docker_mc_export() {
  # Check for docker-compose file
  if ! [ -f "$work_dir/scripts/docker-compose.minio-export.yml" ]; then
    echo "No configuration file for minio migration found!"
    exit 1
  fi

  # Get credentials from the containers
  container="${COMPOSE_PROJECT_PREFIX}-minio-1"
  export AWS_CONNECTION_STRING=$(process_name_to_url "minio") # http://minio:9000
  export AWS_ACCESS_KEY=$(docker exec -i "$container" /bin/sh -c "printenv MINIO_ACCESS_KEY")
  export AWS_SECRET_KEY=$(docker exec -i "$container" /bin/sh -c "printenv MINIO_SECRET_KEY")

  # Create backup folder
  export backup_minio_dir="${backup_dir}/minio"
  mkdir -p "$backup_minio_dir"

  # Perform the backup using toolbox docker image
  export USER="$(echo $UID)"
  envsubst < "$work_dir/scripts/docker-compose.minio-export.yml" > ./job.yml
  COMPOSE_IGNORE_ORPHANS=True docker compose -f ./job.yml -p "${COMPOSE_PROJECT_PREFIX}" up -d
  monitor_minio_success "minio-export"
  COMPOSE_IGNORE_ORPHANS=True docker compose -f ./job.yml -p "${COMPOSE_PROJECT_PREFIX}" down
  rm ./job.yml
}

# Function to process container names into URLs
process_name_to_url() {
  local name="$1"
  # Replace dots with hyphens and prepend 'http://' and append ':9000'
  local url="http://$(echo "$name" | tr '.' '-' | tr '[:upper:]' '[:lower:]'):9000"

  echo "$url"
}

# Function to track the minio migration job
monitor_minio_success() {
  echo "Waiting for minio download..."
  sleep 5
  local name_short="$1"
  local container_name=$(docker ps -a --filter "name=${name_short}" --format "{{.Names}}" | head -n 1)

  if [[ -z "$container_name" ]]; then
      echo "Error: no '$name' container found."
      exit 1
  fi

  local result=$(docker wait "$container_name")
    if [[ "$result" == "0" ]]; then
        echo "Success!"
        docker logs "$container_name"
    else
        echo "Minio download error! Logs:"
        docker logs "$container_name"
    fi
}

# Function to download influxdb and redis data
services_backup() {
  local target_dir="$backup_dir"
  local services=("$COMPOSE_PROJECT_PREFIX-auth-cache-1" "$COMPOSE_PROJECT_PREFIX-influxdb-1")
  echo "Backing up services..."
  for i in "${services[@]}"; do
    start_container "$i" 
  done
  local redis_container=$(docker ps -f name=$COMPOSE_PROJECT_PREFIX --format "{{.Names}}" | grep auth-cache)
  local influx_container=$(docker ps -f name=$COMPOSE_PROJECT_PREFIX --format "{{.Names}}" | grep influx)
  
  cp_from_container "$redis_container" "/data" "${target_dir}/redis" ## redis/data....
  cp_from_container "$influx_container" "/var/lib/influxdb" "${target_dir}/influxdb" ## influxdb/influxdb...
  for i in "${services[@]}"; do
    stop_container "$i" 
  done
}

# Function to copy data from container
cp_from_container() {
  local container="$1"
  local source_path="$2"
  local target_host_path="$3"

  mkdir -p "$target_host_path"
  docker cp "${container}:${source_path}" "$target_host_path"
}

# Function to create a TAR archive
create_tar_archive() {
  local tar_target_dir="$1"
  local tar_filename="$2"
  echo "Creating TAR archive..."
  tar -cf "$tar_filename" -C "$(dirname "$tar_target_dir")" "$(basename "$tar_target_dir")"
  rm -rf "$tar_target_dir"
  echo "TAR archive created successfully."
}

backup_certs() {
  local name_short="$1"
  local container_name=$(docker ps -a --filter "name=${name_short}" --format "{{.Names}}" | head -n 1)
  local certs_dir="$2"

  # Get the mount path of 'trusted-certificates-volume'
  local mount_path=$(docker inspect -f '{{ range .Mounts }}{{.Source}} {{ .Destination }}{{ end }}' "$container_name" | grep trusted-certificates-volume | awk '{print $2}')

  # Copy the certificates to the target directory
  cp_from_container "$container_name" "$mount_path" "$certs_dir"
  echo "Certificates copied."
}

stop_container() {
  local container="$1"
  docker stop "$container" > /dev/null 2>&1
  while [ $(docker ps --format '{{.Names}}' | grep "${COMPOSE_PROJECT_PREFIX}" | grep -c "$container") != 0 ]; do
    sleep 1
  done
}

start_container() {
  local container="$1"
  docker start "$container" > /dev/null 2>&1
  while [ $(docker ps --format '{{.Names}}' | grep "${COMPOSE_PROJECT_PREFIX}" | grep -c "$container") != 1 ]; do
    sleep 1
  done
}

# Main script logic
main() {
  pwd_compatible
  backup_dir="${work_dir}/backup"
  mkdir -p "$backup_dir"

  # Get the container names for PostgreSQL and MiniO
  # instead of containers, get the list of databases to backup
  export postgres_container="$COMPOSE_PROJECT_PREFIX-db-1"
  export db_names=("testitdb" "authdb" "backgrounddb" "avatarsdb" "licensedb" "globalsearchdb")
  export minio_container="$COMPOSE_PROJECT_PREFIX-minio-1"

  # Backup trusted-certificates-volume 
  backup_certs "webapi" "${backup_dir}/certs"

  # Stop the Test IT containers
  docker compose -f "${COMPOSE_FILE_PATH}" -p "${COMPOSE_PROJECT_PREFIX}" stop
  # Perform the backups
  services_backup
  backup_postgres
  backup_minio
  
  create_tar_archive "$backup_dir" "$work_dir/testit_docker_to_k8s.tar"

  # Start the Test IT containers
  docker compose -f "${COMPOSE_FILE_PATH}" -p "${COMPOSE_PROJECT_PREFIX}" up -d

  echo "Backup successful! Archive saved at: '$work_dir/testit_docker_to_k8s.tar'"
  exit 0
}

# Array of required environment variables
minio_vars=("TMS_BUCKET_NAME" "AVATARS_AWS_BUCKET_NAME")
docker_vars=("COMPOSE_FILE_PATH" "COMPOSE_PROJECT_PREFIX" "COMPOSE_NETWORK_NAME")
check_required_variables_array "minio_vars" "${minio_vars[@]}"
check_required_variables_array "Docker Variables" "${docker_vars[@]}"

command_check "docker compose"
command_check "tar"

main "$@"
