#!/bin/bash
# move_to_k8s.sh

set -euo pipefail

# Sources the common-functions.sh script from the directory path of the current script
source "$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"/common-functions.sh

# Function to run k8s job and track its completion
run_job() {
  local job_name="$1"
  local job_path="$2"
  local source="$3"
  local arch="$4"

  # Create archive
  tar -C "$source" -cf "${arch}" .

  envsubst < "${job_path}/job.yaml" > ./tmp.yaml
  kubectl apply -f "${job_path}/configmap.yaml" -n "$namespace" > /dev/null 2>&1
  kubectl apply -f ./tmp.yaml -n "$namespace" > /dev/null 2>&1
  # Get the name of the pod associated with the job
  local pod_name=""
  echo "Waiting for job '$job_name' to start..."
  local status=0
  while [ $status -lt 1 ]; do
    pod_name=$(pod_by_label "job-name=$job_name")
    if [ "$(kubectl get pods -n "$namespace" "$pod_name" -o jsonpath='{.status.containerStatuses[0].ready}')" == "true" ]; then
      status=$(kubectl logs "$pod_name" -n "$namespace" | grep -c "Waiting")
    fi
    sleep 1
  done
  # Copy backup files to pod
  copy_to_pod "$pod_name" "$arch" "/tmp/archive.tar"
  echo "Init" > "${buffer_dir}/start.txt"
  copy_to_pod "$pod_name" "${buffer_dir}/start.txt" "/tmp/start.txt"
  # Wait for the job to complete or fail using the loop with sleep
  while true; do
    local succeeded_status=$(kubectl logs "$pod_name" -n "$namespace" | grep -c "Script finished.")
    local failed_status=$(kubectl get job "$job_name" -n "$namespace" -o=jsonpath='{.status.failed}')
    
    if [ "$succeeded_status" = "1" ]; then
      echo "Job '$job_name' successful."
      # Display the logs of the job
      kubectl logs "$pod_name" -n "$namespace" | grep -v "Waiting"
      kubectl delete -f ./tmp.yaml -n "$namespace" > /dev/null 2>&1
      kubectl delete -f "${job_path}/configmap.yaml" -n "$namespace" > /dev/null 2>&1
      rm -f ./tmp.yaml
      break
    elif [ "$failed_status" = "1" ]; then
      echo "Job '$job_name' failed. Displaying logs:"
      # Display the logs of the job
      kubectl logs "$pod_name" -n "$namespace"
      kubectl delete -f ./tmp.yaml -n "$namespace" > /dev/null 2>&1
      kubectl delete -f "${job_path}/configmap.yaml" -n "$namespace" > /dev/null 2>&1
      rm -f ./tmp.yaml
      exit 1
    fi
    sleep 1
  done
}

# Function to upload Redis and InfluxDB data
upload_redis_influx() {
  echo "Uploading Redis + InfluxDB..."
  local influxdb_arch="${buffer_dir}/influxdb.tar"
  local redis_dir="${buffer_dir}/backup/redis"
  local influxdb_dir="${buffer_dir}/backup/influxdb"
  check_directory_existence "$redis_dir"
  check_directory_existence "$influxdb_dir"

  local redis_pod=$(pod_by_label "app=auth-cache")
  local influx_pod=$(pod_by_label "app=influxdb")

  # Copy backup files for redis
  for file in $(ls "$redis_dir"); do
    local full_path="${redis_dir}/${file}"
    kubectl cp "$full_path" "$redis_pod:/data/${file}" -n "$namespace" --retries 999 --no-preserve
  done

  tar -C "$influxdb_dir" -cf "$influxdb_arch" .

  # Copy backup archives to influxdb pod and unpack
  kubectl cp "$influxdb_arch" "${influx_pod}:tmp/influxdb.tar" -n "$namespace" --retries 999 --no-preserve
  kubectl exec "$influx_pod" -n "$namespace" -- tar -C var/lib/influxdb -xf tmp/influxdb.tar
  kubectl exec "$influx_pod" -n "$namespace" -- rm -f tmp/influxdb.tar

  echo "Data restored to 'auth-cache' and 'influxdb'."
}

# Function to upload to minio server form docker backups
upload_minio() {
  echo "Uploading buckets..."
  local minio_dir="$buffer_dir/backup/minio"
  check_directory_existence "${minio_dir}/testit"
  check_directory_existence "${minio_dir}/avatars"
  
  run_job "mc-upload-job" "./jobs/minio/docker-to-k8s" "$minio_dir" "${buffer_dir}/minio.tar"
}

# Function to restore databases from docker backups
upload_postgres() {
  echo "Uploading databases..."
  local postgres_dir="$buffer_dir/backup/postgres"
  check_directory_existence "$postgres_dir"
  local databases=("testitdb" "authdb" "avatarsdb" "backgrounddb" "licensedb" "globalsearchdb")
  for db in "${databases[@]}"; do
    check_file_existence "$postgres_dir/${db}_backup.sql"
  done

  run_job "postgres-export-job" "./jobs/postgres/docker-to-k8s" "$postgres_dir" "${buffer_dir}/postgres.tar"
}

main () {
  # Main script logic
  echo "Starting restore process for namespace '$namespace' from docker backup..."

  pwd_compatible
  export buffer_dir="${work_dir}/buffer"
  mkdir -p "$buffer_dir"
  local archive="$2"
  tar -xf "$archive" -C "$buffer_dir" 

  # Save unpacked trusted certificates
  rm -rf "$work_dir/CERTS/"
  mkdir -p "$work_dir/CERTS/"
  mv "${buffer_dir}/backup/certs/ca-certificates/" "$work_dir/CERTS/" 
  echo "Trusted certificates saved at '$work_dir/CERTS/'."

  # Stop deployments
  declare -A replicas
  export replicas
  local testit=("frontend" "webapi" "auth" "avatars-api" "background-service" "ldapwebapi" "license-service" "globalsearch-service" )
  echo "Stopping Test IT..."
  stop_deployments_save_replicas "${testit[@]}"

  # Get minio creds
  export AWS_ACCESS_KEY=$(get_env_variable_from_pod "minio" "MINIO_ROOT_USER")
  export AWS_SECRET_KEY=$(get_env_variable_from_pod "minio" "MINIO_ROOT_PASSWORD")
  upload_minio 

  # Get posrtges creds
  export POSTGRES_USER=$(get_env_variable_from_pod "postgres" "POSTGRES_USER")
  export POSTGRES_PASSWORD=$(get_env_variable_from_pod "postgres" "POSTGRES_PASSWORD")
  upload_postgres

  upload_redis_influx

  rm -rf "$buffer_dir"

  # Start deployments
  echo "Starting Test IT..."
  start_deployments_from_saved_replicas "${testit[@]}"

  echo "Restore process completed successfully! Your Test IT is running on Kubernetes now!"
  echo "If you have SSL certificates (trusted-certificates-volume) or any other configurations to apply, you can reference the Test IT docs for guides."
  exit 0
}

command_check "kubectl"
command_check "tar"

# Array of required environment variables
minio_vars=("AWS_CONNECTION_STRING" "TESTIT_BUCKET" "AVATARS_BUCKET")
postgres_vars=("POSTGRES_HOST" "POSTGRES_PORT")
check_required_variables_array "minio_vars" "${minio_vars[@]}"
check_required_variables_array "postgres_vars" "${postgres_vars[@]}"
export USAGE_MESSAGE="Usage: $0 <namespace> <archive>"
validate_input "$@" 2
namespace="$1"
validate_namespace "$namespace"
main "$@"
