#!/bin/bash
# k8s_restore.sh

set -euo pipefail

# Sources the common-functions.sh script from the directory path of the current script
source "$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"/common-functions.sh

# Function to run k8s job and track its completion
run_job() {
  local job_name="$1"
  local job_path="$2"
  local arch="$3"
  envsubst < "${job_path}/job.yaml" > ./tmp.yaml
  kubectl apply -f "${job_path}/configmap.yaml" -n "$namespace" > /dev/null 2>&1
  kubectl apply -f ./tmp.yaml -n "$namespace" > /dev/null 2>&1
  # Get the name of the pod associated with the job
  local pod_name=""
  local status=0
  while [ $status -lt 1 ]; do
    pod_name=$(pod_by_label "job-name=$job_name")
    if [ "$(kubectl get pods -n "$namespace" "$pod_name" -o jsonpath='{.status.containerStatuses[0].ready}')" == "true" ]; then
      status=$(kubectl logs "$pod_name" -n "$namespace" | grep -c "Waiting")
    fi
    sleep 1
  done
  # Copy backup files to pod
  copy_to_pod "$pod_name" "$arch" "/tmp/archive.tar"
  echo "Init" > start.txt
  copy_to_pod "$pod_name" "start.txt" "/tmp/start.txt"
  rm -f start.txt
  # Wait for the job to complete or fail using the loop with sleep
  while true; do
    local succeeded_status=$(kubectl logs "$pod_name" -n "$namespace" 2>/dev/null | grep -c "Script finished.")
    local failed_status=$(kubectl get job "$job_name" -n "$namespace" -o=jsonpath='{.status.failed}')
    
    if [ "$succeeded_status" = "1" ]; then
      echo "Job '$job_name' successful."
      kubectl logs "$pod_name" -n "$namespace" | grep -v "Waiting"
      kubectl delete -f ./tmp.yaml -n "$namespace" 2>&1
      kubectl delete -f "${job_path}/configmap.yaml" -n "$namespace" > /dev/null 2>&1
      rm -f ./tmp.yaml
      break
    elif [ "$failed_status" = "1" ]; then
      echo "Job '$job_name' failed. Displaying logs:"
      # Display the logs of the failed pod
      kubectl logs "$pod_name" -n "$namespace"
      kubectl delete -f ./tmp.yaml -n "$namespace" 2>&1
      kubectl delete -f "${job_path}/configmap.yaml" -n "$namespace" > /dev/null 2>&1
      rm -f ./tmp.yaml
      exit 1
    fi
    sleep 1
  done
}

# Function to restore services from backup
restore_services() {
  local redis_arch="${backup_path}/redis_${arch_postfix}"
  local influx_arch="${backup_path}/influxdb_${arch_postfix}"
  check_file_existence "$redis_arch"
  check_file_existence "$influx_arch"

  local redis_pod=$(pod_by_label "app=auth-cache")
  local influx_pod=$(pod_by_label "app=influxdb")

  # Unpack backup archives and copy to redis pod
  rm -rf "${backup_path}"/redis-tmp || true
  mkdir "${backup_path}"/redis-tmp
  tar -C "${backup_path}"/redis-tmp -xf "$redis_arch"
  for file in $(ls ${backup_path}/redis-tmp); do
    local full_path="${backup_path}/redis-tmp/$file"
    kubectl cp "$full_path" "$redis_pod:/data/$file" -n "$namespace" --retries 999 --no-preserve
  done
  rm -rf "${backup_path}"/redis-tmp

  # Copy backup archives to influxdb pod and unpack
  kubectl cp "$influx_arch" "$influx_pod:/tmp/influxdb.tar" -n "$namespace" --retries 999 --no-preserve
  kubectl exec "$influx_pod" -n "$namespace" -- tar -C /var/lib/influxdb -xf /tmp/influxdb.tar
  kubectl exec "$influx_pod" -n "$namespace" -- rm -f /tmp/influxdb.tar

  echo "Data restored to 'auth-cache' and 'influxdb'."
}

restore_postgres() {
  local target_arch="${backup_path}/postgres_${arch_postfix}"
  check_file_existence "$target_arch"

  # Run restore job
  run_job "postgres-restore-job" "./jobs/postgres/restore" "$target_arch"

  echo "Postgres databases restored."
}

restore_minio() {
  local target_arch="${backup_path}/minio_${arch_postfix}"
  check_file_existence "$target_arch"

  # Run restore job
  run_job "minio-restore-job" "./jobs/minio/restore" "$target_arch"
  
  echo "MiniO restored."
}

main() {
  # Main script logic
  echo "Starting restore process for namespace '$namespace' from backup date '$backup_date'..."

  # Check for installed charts in a namespace
  validate_testit_charts_presence "$namespace" 1

  export arch_postfix="backup.tar"
  pwd_compatible
  export backup_path="${work_dir}/backups/${backup_date}"
  check_directory_existence "$backup_path"

  declare -A replicas
  export replicas
  # Stop deployments
  echo "Stopping TestIT..."
  local testit=("frontend" "webapi" "background-service" "auth" "avatars-api" "ldapwebapi" "license-service" "globalsearch-service")
  stop_deployments_save_replicas "${testit[@]}"

  # Restore services
  echo "Restoring services..."
  restore_services

  # Restore databases
  echo "Restoring databases..."
  restore_postgres

  # Restore minio
  echo "Restoring buckets..."
  restore_minio

  # Start deployments
  echo "Starting TestIT..."
  start_deployments_from_saved_replicas "${testit[@]}"

  echo "Restore process completed successfully!"
  exit 0
}

# Check for required environment variables
postgres_vars=("POSTGRES_HOST" "POSTGRES_PORT" "POSTGRES_USER" "POSTGRES_PASSWORD")
minio_vars=("AWS_CONNECTION_STRING" "AWS_ACCESS_KEY" "AWS_SECRET_KEY")
check_required_variables_array "postgres_vars" "${postgres_vars[@]}"
check_required_variables_array "minio_vars" "${minio_vars[@]}"
command_check "kubectl"
command_check "helm"
command_check "tar"
export USAGE_MESSAGE="Usage: $0 <namespace> DD_MM_YYYY"
validate_input "$@" 2
namespace="$1"
validate_namespace "$namespace"
backup_date="$2"

main "$@"
exit 0
