#!/bin/bash
# k8s_postgres_reindex_alpine.sh

set -euo pipefail

# Sources the common-functions.sh script from the directory path of the current script
source "$(dirname "${BASH_SOURCE[0]}")"/common-functions.sh

main() {
  # Main script logic
  # Start message
  echo "Starting Database reindex for namespace '$namespace'..."

  # Check for installed charts in a namespace
  validate_testit_charts_presence "$namespace" 1

  # Create array for saving replicaCount
  declare -A replicas
  export replicas

  # Stop TestIT deployments
  local deployments=("license-service" "auth" "ldapwebapi" "background-service" "rabbitmqconsumer" "avatars-api" "globalsearch-service" "webapi" "frontend")
  stop_deployments_save_replicas "${deployments[@]}"

  # Get a name of the PostgreSQL pod
  local pod=$(pod_by_label "app=postgres")
  local psql_user=$(get_env_variable_from_pod "postgres" "POSTGRES_USER")

  # Execute reindex for databases
  local databases=("testitdb" "licensedb" "authdb" "globalsearchdb" "backgrounddb" "avatarsdb")
  for database in "${databases[@]}"; do
    execute_psql "$pod" "$psql_user" "$database" "reindex database $database;"
  done

  # Start deployments
  echo "Starting TestIT..."
  start_deployments_from_saved_replicas "${deployments[@]}"

  # Success message
  echo "Reindex completed successfully!"
  exit 0
}

# Validations before the script
validate_input "$@" 1
command_check "kubectl"
command_check "helm"
namespace="$1"
validate_namespace "$namespace"

main "$@"
exit 0
