#!/bin/bash
# k8s_postgres_migrate.sh 

set -euo pipefail

# Default values
export VERBOSE="false"
cleanup=false

# Function to display flags usage
usage() {
  echo "Usage: $0 [-d] [-c] <namespace>" >&2
  echo "  -d  Enable debug mode"
  echo "  -c  Perform cleanup"
  exit 1
}

# Process command-line options
while getopts ":dc" opt; do
  case $opt in
    d)
      export VERBOSE="true"
      set -x  # Enable debug mode
      ;;
    c)
      cleanup=true
      ;;
    \?)
      echo "Invalid option: -$OPTARG" >&2
      usage
      exit 1
      ;;
  esac
done
# Remove the processed options from the argument list
shift $((OPTIND - 1))

# Array of required environment variables
required_vars=("DUMP_VOLUME_SIZE_GB")

# Function to check if variables are set and not empty
check_required_variables() {
  local unset_vars=()
  for var in "${required_vars[@]}"; do
    if [[ -z "${!var:-}" ]]; then
      unset_vars+=("$var")
    fi
  done

  if [[ ${#unset_vars[@]} -gt 0 ]]; then
    echo "Please set the following environment variables: ${unset_vars[*]}"
    exit 1
  fi
}

# Check for required dependencies
command_check() {
  local required_command="$1"
  command -v "$required_command" >/dev/null 2>&1 || {
    echo "'$required_command' command not found. Make sure it's installed and in the PATH."
    exit 1
  }
}

# Validate input
validate_input() {
  if [ $# -ne 1 ]; then
    echo "Usage: $0 <namespace>"
    exit 1
  fi
}

# Validate the existence of the specified namespace
validate_namespace() {
    local namespace="$1"

    # Check if the provided namespace exists
    if ! kubectl get namespace "$namespace" &> /dev/null; then
        echo "Namespace '$namespace' not found."
        exit 1
    fi
}

# Compatibility for Windows
pwd_compatible(){
  if [[ "$(expr substr $(uname -s) 1 5)" == "MINGW" ]]; then
    work_dir=$(pwd -W)
  else
    work_dir=$(pwd)
  fi
}

# Function to stop deployments
stop_deployments() {
  local s_deployments=("$@")

  for deployment in "${s_deployments[@]}"; do
    echo "Stopping deployment '$deployment'..."
    kubectl scale deployment "$deployment" -n "$namespace" --replicas 0 > /dev/null 2>&1
  done
  for deployment in "${s_deployments[@]}"; do
    while ! [ -z "$(kubectl get deployment $deployment -n $namespace -o=jsonpath='{.status.readyReplicas}')" ]; do
      sleep 1
    done
  done
}

# Function to update deployments image
update_deployments_image() {
  local image="$1"
  shift
  local u_deployments=("$@")
  stop_deployments "${u_deployments[@]}"
  for deployment in "${u_deployments[@]}"; do
    echo "Updating deployment ${deployment} with image ${image}..."
    local container=$(kubectl get deployment $deployment -n $namespace -o=jsonpath='{.spec.template.spec.containers[0].name}')
    kubectl set image deployment/$deployment $container=$image -n $namespace
  done
}

# Function to start deployments
start_deployments() {
  local s_deployments=("$@")
  for deployment in "${s_deployments[@]}"; do
    kubectl scale deployment "$deployment" -n "$namespace" --replicas 1 > /dev/null 2>&1
  done

  local counter=0
  local timeout=180
  for deployment in "${s_deployments[@]}"; do
    while [[ -z "$(kubectl get deployment "$deployment" -n "$namespace" -o=jsonpath='{.status.readyReplicas}')" ]]; do
      sleep 1
      ((counter+=1))
      if [[ "$counter" == "$timeout" ]]; then
        echo "Deployment '$deployment' is taking longer than ${timeout}s to become ready."
        break
      fi
    done
  done
}

# Function to run k8s job and track its completion
run_job() {
  local job_name="$1"
  local job_path="$2"
  envsubst < "${job_path}/job.yaml" > ./tmp.yaml
  kubectl apply -f "${job_path}/configmap.yaml" -n "$namespace" > /dev/null 2>&1
  kubectl apply -f ./tmp.yaml -n "$namespace" > /dev/null 2>&1

  # Wait for the job to complete or fail using the loop with sleep
  while true; do
    # Get the name of the pod associated with the job
    local pod_name=$(kubectl get pods -n "$namespace" -l job-name="$job_name" -o jsonpath='{.items[0].metadata.name}')
    local succeeded_status=$(kubectl logs "$pod_name" -n "$namespace" | grep -c "Script finished.")
    local failed_status=$(kubectl get job "$job_name" -n "$namespace" -o=jsonpath='{.status.failed}')
    
    if [ "$succeeded_status" -ge "1" ]; then
      echo "Job '$job_name' successful."
      # Save the logs of the pod
      kubectl logs "$pod_name" -n "$namespace" &> "${PWD}/$job_name-$(date +"%d-%m-%Y")-success.log"
      kubectl delete -f ./tmp.yaml -n "$namespace" 2>&1
      kubectl delete -f "${job_path}/configmap.yaml" -n "$namespace" > /dev/null 2>&1
      if $cleanup; then rm -f ./tmp.yaml; fi
      break
    elif [ "$failed_status" = "1" ]; then
      # Save the logs of the pod
      kubectl logs "$pod_name" -n "$namespace" &> "${PWD}/$job_name-fail.log"
      echo "Job '$job_name' failed. Logs: '${PWD}/$job_name-$(date +"%d-%m-%Y")-fail.log'."
      kubectl delete -f ./tmp.yaml -n "$namespace" 2>&1
      kubectl delete -f "${job_path}/configmap.yaml" -n "$namespace" > /dev/null 2>&1
      if $cleanup; then rm -f ./tmp.yaml; fi
      exit 1
    fi
    sleep 15
  done
}

# Delete PersistentVolumeClaims, saving configs for recreation
delete_pvc() {
  local d_deployments=("db" "authdb" "avatars-db")
  echo "Deleting PVC for ${d_deployments[@]}"
  mkdir -p "$backup_dir/pvc_configs"
  local pvc_name
  for deployment in "${d_deployments[@]}"
  do
    pvc_name=$(kubectl -n "$namespace" get pvc -l app=$deployment -ojsonpath='{.items[0].metadata.name}')
    check=$(kubectl get pvc -n $namespace $pvc_name --no-headers 2>/dev/null | wc -l)
    if [ $check -ne 1 ]; then
      echo "Error! No PVC '$pvc_name' found."
      exit 1
    else
      # Save the PVC YAML configuration to a file
      kubectl get pvc $pvc_name -n "$namespace" -o yaml > "$backup_dir/pvc_configs/${pvc_name}.yaml"
      # Remove kubernetes-generated fields
      trim_pvc_yaml "$backup_dir/pvc_configs/${pvc_name}.yaml"
      echo "Saved PVC configuration for '$pvc_name' to '$backup_dir/pvc_configs/${pvc_name}.yaml'."
    fi
  done
  stop_deployments "${d_deployments[@]}"
  # Delete PVCs
  kubectl delete -f "$backup_dir/pvc_configs/"
}

trim_pvc_yaml () {
  local file="$1"
  yq eval 'del(.metadata.annotations) | del(.metadata.creationTimestamp) | del(.metadata.finalizers) | del(.metadata.resourceVersion) | del(.metadata.uid) | del(.spec.volumeName) | del(.status)' "$file" > "$file.tmp"
  mv "$file.tmp" "$file"
}

# Recreate PersistentVolumeClaims using saved configs from 'delete_pvc()'
recreate_pvc() {
  local pvc_configs_dir="$backup_dir/pvc_configs"
  local pvc_files=("$pvc_configs_dir"/*.yaml)
  if [ ${#pvc_files[@]} -eq 0 ]; then
    echo "No PVC configurations found in the backup directory: $pvc_configs_dir"
    exit 1
  fi

  echo "Recreating PVCs from the backup directory: $pvc_configs_dir"
  for pvc_file in "${pvc_files[@]}"; do
    # Extract the PVC name from the filename (remove the directory and extension)
    pvc_name=$(basename "$pvc_file" .yaml)

    # Recreate the PVC from the saved YAML configuration
    kubectl apply -f "$pvc_file" > /dev/null 2>&1

    # Wait until the PVC is created
    while [ -z "$(kubectl get pvc "$pvc_name" -n "$namespace" -o 'jsonpath={.status.phase}')" ]; do
      sleep 1
    done

    echo "Recreated PVC '$pvc_name'."
  done

  echo "PVC recreation complete."
}

# Create dumps for the databases
backup_dbs() {
  export DUMP_VOLUME_STORAGE_CLASS=$(kubectl -n "$namespace" get pvc -l app=db -ojsonpath='{.items[0].spec.storageClassName}')
  export index=0
  export dumps_pvc_name="psql-dumps-$(date +"%d-%m-%Y")-$index"
  while [ $(kubectl -n $namespace get pvc $dumps_pvc_name --no-headers 2>/dev/null | wc -l ) -ne 0 ]; do
    ((index+=1))
    export dumps_pvc_name="psql-dumps-$(date +"%d-%m-%Y")-$index"
  done
  envsubst < ./jobs/postgres/import/pvc.yaml > $work_dir/$dumps_pvc_name.yaml
  kubectl -n $namespace apply -f $work_dir/$dumps_pvc_name.yaml >/dev/null
  run_job "postgres-import-job" "./jobs/postgres/import"
}

# Restore databases from dumps
restore_dbs() {
  run_job "postgres-export-job" "./jobs/postgres/export"
  if $cleanup; then kubectl -n $namespace delete -f $work_dir/$dumps_pvc_name.yaml; fi
  if $cleanup; then rm -f $work_dir/$dumps_pvc_name.yaml; fi
}

main() {
  namespace="$1"
  validate_namespace "$namespace"

  # Create local buffer
  pwd_compatible
  backup_dir="$work_dir/temp"
  if $cleanup; then rm -rf "$backup_dir"; fi
  if ! [ -d "$backup_dir" ]; then mkdir -p "$backup_dir"; fi

  # Stop all the Test IT deployments except database servers
  local deployments=("frontend" "webapi" "rabbitmq" "rabbitmqconsumer" "auth" "ldapwebapi" "auth-cache" "license-service" "avatars-api" "influxdb" "transfer-service")
  stop_deployments "${deployments[@]}"

  # Import data
  local databases=("db" "authdb" "avatars-db" "backgrounddb")
  start_deployments "${databases[@]}"
  export POSTGRES_PORT="$(kubectl -n $namespace get svc -l app=db -ojsonpath='{.items[0].spec.ports[0].port}')"
  export PGHOST_1="$(kubectl -n $namespace get svc -l app=db -ojsonpath='{.items[0].metadata.name}')"
  export PGHOST_2="$(kubectl -n $namespace get svc -l app=authdb -ojsonpath='{.items[0].metadata.name}')"
  export PGHOST_3="$(kubectl -n $namespace get svc -l app=avatars-db  -ojsonpath='{.items[0].metadata.name}')"
  backup_dbs

  # Update PostgreSQL deployments
  delete_pvc
  recreate_pvc
  local new_image="postgres:14.7-bullseye"
  update_deployments_image "$new_image" "${databases[@]}"

  # Export data
  start_deployments "${databases[@]}"
  restore_dbs
  # Cleanup
  if $cleanup; then rm -rf "$backup_dir"; fi
  
  echo "Migration successful! Restarting Test IT..."
  start_deployments "${deployments[@]}"
}

# Validations before the script
check_required_variables
validate_input "$@"
command_check "kubectl"
command_check "yq"

main "$@"