#!/bin/bash
# k8s_minio_migrate.sh

set -euo pipefail

# Default values
export VERBOSE="false"
cleanup=false

# Function to display flags usage
usage() {
  echo "Usage: $0 [-d] [-c] <namespace>" >&2
  echo "  -d  Enable debug mode"
  echo "  -c  Perform cleanup"
  exit 1
}

# Process command-line options
while getopts ":dc" opt; do
  case $opt in
    d)
      export VERBOSE="true"
      set -x  # Enable debug mode
      ;;
    c)
      cleanup=true
      ;;
    \?)
      echo "Invalid option: -$OPTARG" >&2
      usage
      exit 1
      ;;
  esac
done
# Remove the processed options from the argument list
shift $((OPTIND - 1))

# Check for required dependencies
command_check() {
  local required_command="$1"
  command -v "$required_command" >/dev/null 2>&1 || {
    echo "'$required_command' command not found. Make sure it's installed and in the PATH."
    exit 1
  }
}

# Validate input
validate_input() {
  if [ $# -ne 1 ]; then
    usage
    exit 1
  fi
}

# Validate the existence of the specified namespace
validate_namespace() {
    local namespace="$1"

    # Check if the provided namespace exists
    if ! kubectl get namespace "$namespace" &> /dev/null; then
        echo "Namespace '$namespace' not found."
        exit 1
    fi
}

# Compatibility for Windows
pwd_compatible() {
  if [[ "$(expr substr $(uname -s) 1 5)" == "MINGW" ]]; then
    work_dir=$(pwd -W)
  else
    work_dir=$(pwd)
  fi
}

# Function to stop deployments
stop_deployments() {
  local s_deployments=("$@")

  for deployment in "${s_deployments[@]}"; do
    echo "Stopping deployment '$deployment'..."
    kubectl scale deployment "$deployment" -n "$namespace" --replicas 0 > /dev/null 2>&1
  done
  for deployment in "${s_deployments[@]}"; do
    while ! [ -z "$(kubectl get deployment $deployment -n $namespace -o=jsonpath='{.status.readyReplicas}')" ]; do
      sleep 1
    done
  done
}

# Function to create new pvc with same configs
create_new_pvc() {
  export tag="$1"
  old_pvc_name=$(kubectl -n "$namespace" get pvc -l app=$tag -ojsonpath='{.items[0].metadata.name}')
  export index=0
  export new_pvc_name="$tag-restored-$(date +"%d-%m-%Y")-$index"
  while [ $(kubectl -n $namespace get pvc $new_pvc_name --no-headers 2>/dev/null | wc -l ) -ne 0 ]; do
    ((index+=1))
    export new_pvc_name="$tag-restored-$(date +"%d-%m-%Y")-$index"
  done
  export new_pvc_access=$(kubectl -n "$namespace" get pvc $old_pvc_name -ojsonpath='{.status.accessModes[0]}')
  export new_pvc_size=$(kubectl -n "$namespace" get pvc $old_pvc_name -ojsonpath='{.status.capacity.storage}')
  export new_pvc_storageclass=$(kubectl -n "$namespace" get pvc $old_pvc_name -ojsonpath='{.spec.storageClassName}')
  envsubst < "./jobs/minio/migrate/pvc.yaml" > ${work_dir}/$new_pvc_name.yaml
  kubectl -n "$namespace" apply -f ${work_dir}/$new_pvc_name.yaml >/dev/null
  echo "$new_pvc_name"
}

# Function to run k8s job and track its completion
run_job() {
  local job_name="$1"
  local job_path="$2"
  envsubst < "${job_path}/job.yaml" > ./tmp.yaml
  kubectl apply -f "${job_path}/configmap.yaml" -n "$namespace" > /dev/null 2>&1
  kubectl apply -f ./tmp.yaml -n "$namespace" > /dev/null 2>&1
  # Wait for the job to complete or fail using the loop with sleep
  while true; do
    # Get the name of the pod associated with the job
    local pod_name=$(kubectl get pods -n "$namespace" -l job-name="$job_name" -o jsonpath='{.items[0].metadata.name}')
    local succeeded_status=$(kubectl logs "$pod_name" -n "$namespace" | grep -c "Script finished.")
    local failed_status=$(kubectl get job "$job_name" -n "$namespace" -o=jsonpath='{.status.failed}')
    
    if [ "$succeeded_status" -ge "1" ]; then
      echo "Job '$job_name' successful."
      # Save the logs of the pod
      kubectl logs "$pod_name" -n "$namespace" &> "${PWD}/$job_name-$(date +"%d-%m-%Y")-success.log"
      kubectl delete -f ./tmp.yaml -n "$namespace" 2>&1
      kubectl delete -f "${job_path}/configmap.yaml" -n "$namespace" > /dev/null 2>&1
      if $cleanup; then rm -f ./tmp.yaml; fi
      break
    elif [ "$failed_status" = "1" ]; then
      # Save the logs of the pod
      kubectl logs "$pod_name" -n "$namespace" &> "${PWD}/$job_name-$(date +"%d-%m-%Y")-fail.log"
      echo "Job '$job_name' failed. Logs: '${PWD}/$job_name-$(date +"%d-%m-%Y")-fail.log'."
      kubectl delete -f ./tmp.yaml -n "$namespace" 2>&1
      kubectl delete -f "${job_path}/configmap.yaml" -n "$namespace" > /dev/null 2>&1
      if $cleanup; then rm -f ./tmp.yaml; fi
      exit 1
    fi
    sleep 15
  done
}

# Function to update deployments image
update_deployments_image() {
  local image="$1"
  shift
  local u_deployments=("$@")
  for deployment in "${u_deployments[@]}"; do
    echo "Updating deployment ${deployment} with image ${image}..."
    local container=$(kubectl get deployment $deployment -n $namespace -o=jsonpath='{.spec.template.spec.containers[0].name}')
    kubectl -n $namespace scale deployment $deployment --replicas 0 > /dev/null 2>&1
    kubectl set image deployment/$deployment $container=$image -n $namespace
  done
}

update_deployment_pvc() {
  local deployment="$1"
  local pvc_name="$2"
  local volume_name=$(kubectl -n $namespace get deploy $deployment -ojsonpath='{.spec.template.spec.volumes[0].name}')
  kubectl -n $namespace patch deployment $deployment -p "{\"spec\":{\"template\":{\"spec\":{\"volumes\":[{\"name\":\"$volume_name\",\"persistentVolumeClaim\":{\"claimName\":\"$pvc_name\"}}]}}}}"
  kubectl -n $namespace scale deployment $deployment --replicas 1 > /dev/null 2>&1
  local counter=0
  local timeout=180
  while [[ -z "$(kubectl get deployment "$deployment" -n "$namespace" -o=jsonpath='{.status.readyReplicas}')" ]]; do
    sleep 1
    ((counter+=1))
    if [[ "$counter" == "$timeout" ]]; then
      echo "Deployment '$deployment' is taking longer than ${timeout}s to become ready."
      break
    fi
  done
}

main() {
  namespace="$1"
  validate_namespace "$namespace"

  # Create local buffer
  pwd_compatible
  
  # Stop all the Test IT deployments except minio servers
  local deployments=("frontend" "webapi" "rabbitmq" "rabbitmqconsumer" "auth" "ldapwebapi" "auth-cache" "license-service" "avatars-api" "influxdb" "transfer-service")
  stop_deployments "${deployments[@]}"

  export minio_pvc_name=$(create_new_pvc "minio")
  export avatars_pvc_name=$(create_new_pvc "avatars-minio")

  # Migrate data from buckets
  export AWS_CONNECTION_STRING="http://$(kubectl -n $namespace get svc -l app=minio -ojsonpath='{.items[0].metadata.name}'):$(kubectl -n $namespace get svc -l app=minio -ojsonpath='{.items[0].spec.ports[0].port}')"
  export AVATARS_AWS_CONNECTION_STRING="http://$(kubectl -n $namespace get svc -l app=avatars-minio -ojsonpath='{.items[0].metadata.name}'):$(kubectl -n $namespace get svc -l app=avatars-minio -ojsonpath='{.items[0].spec.ports[0].port}')"
  run_job "mc-migrate-job" "./jobs/minio/migrate"

  # Update Minio deployments
  local buckets=("minio" "avatars-minio")
  local new_image="minio/minio:RELEASE.2023-04-20T17-56-55Z"
  update_deployments_image "$new_image" "${buckets[@]}"
  update_deployment_pvc "minio" "$minio_pvc_name"
  update_deployment_pvc "avatars-minio" "$avatars_pvc_name"
  
  echo "Migration successful! If all checks pass, make sure to delete the old PVC later and update deployments and persistentVolumeClaims in future yamls."
  kubectl -n $namespace get pvc -l app=minio --sort-by .metadata.creationTimestamp
  kubectl -n $namespace get pvc -l app=avatars-minio --sort-by .metadata.creationTimestamp
}

# Validations before the script
validate_input "$@"
command_check "kubectl"
main "$@"