#!/bin/bash
# k8s_backup.sh

set -euo pipefail

# Check for required environment variables
postgres_vars=("POSTGRES_HOST" "POSTGRES_PORT" "POSTGRES_USER" "POSTGRES_PASSWORD")
minio_vars=("AWS_CONNECTION_STRING" "AWS_ACCESS_KEY" "AWS_SECRET_KEY")

# Sources the common-functions.sh script from the directory path of the current script
source "$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"/common-functions.sh

# Function to run k8s job and track its completion
run_job() {
  local job_name="$1"
  local job_path="$2"
  local dest="$3"
  local arch="$4"
  envsubst < "${job_path}/job.yaml" > ./tmp.yaml
  kubectl apply -f "${job_path}/configmap.yaml" -n "$namespace" > /dev/null 2>&1
  kubectl apply -f ./tmp.yaml -n "$namespace" > /dev/null 2>&1

  # Wait for the job to complete or fail using the loop with sleep
  while true; do
    # Get the name of the pod associated with the job
    local pod_name=$(pod_by_label "job-name=$job_name")
    local succeeded_status=$(kubectl logs "$pod_name" -n "$namespace" 2>/dev/null | grep -c "Script finished.")
    local failed_status=$(kubectl get job "$job_name" -n "$namespace" -o=jsonpath='{.status.failed}')
    
    if [ "$succeeded_status" = "1" ]; then
      echo "Job '$job_name' successful."
      copy_arch_from_pod "$pod_name" "$arch" "$dest"
      kubectl logs "$pod_name" -n "$namespace"
      kubectl delete -f ./tmp.yaml -n "$namespace" 2>&1
      kubectl delete -f "${job_path}/configmap.yaml" -n "$namespace" > /dev/null 2>&1
      rm -f ./tmp.yaml
      break
    elif [ "$failed_status" = "1" ]; then
      echo "Job '$job_name' failed. Displaying logs:"
      # Display the logs of the failed pod
      kubectl logs "$pod_name" -n "$namespace"
      kubectl delete -f ./tmp.yaml -n "$namespace" 2>&1
      kubectl delete -f "${job_path}/configmap.yaml" -n "$namespace" > /dev/null 2>&1
      rm -f ./tmp.yaml
      exit 1
    fi
    sleep 1
  done
}

# Function to copy data from pod
copy_arch_from_pod() {
  local pod_name="$1"
  local arch="$2"
  local dest="$3"
  kubectl cp "${pod_name}:/tmp/${arch}" "${dest}/${arch}" -n "$namespace" --retries 999 1>/dev/null 2>&1
  kubectl exec "$pod_name" -n "$namespace" -- rm -rf tmp/"$arch"
}

backup_postgres() {
  local target_arch="postgres_${arch_name}"
  # Run backup job
  run_job "postgres-backup-job" "./jobs/postgres/backup" "$backup_path" "$target_arch" 
  echo "Backup for postgres created in $target_arch."
}

backup_minio() {
  local target_arch="minio_${arch_name}"
  # Run backup job
  run_job "minio-backup-job" "./jobs/minio/backup" "$backup_path" "$target_arch" 
  echo "Backup for minio created in $target_arch."
}

# Function to backup services
backup_services() {
  local redis_arch="${backup_path}/redis_${arch_name}"
  local influx_arch="${backup_path}/influxdb_${arch_name}"

  local redis_pod=$(pod_by_label "app=auth-cache")
  local influx_pod=$(pod_by_label "app=influxdb")

  # Copy backup from redis pod
  kubectl exec "$redis_pod" -n "$namespace" -- tar -C /data -cf /tmp/redis.tar .
  kubectl cp "$redis_pod:/tmp/redis.tar" "$redis_arch" -n "$namespace" --retries 999
  kubectl exec "$redis_pod" -n "$namespace" -- rm -f /tmp/redis.tar

  # Copy backup from influxdb pod
  kubectl exec "$influx_pod" -n "$namespace" -- tar -C /var/lib/influxdb -cf /tmp/influxdb.tar .
  kubectl cp "$influx_pod:/tmp/influxdb.tar" "$influx_arch" -n "$namespace" --retries 999
  kubectl exec "$redis_pod" -n "$namespace" -- rm -f /tmp/influxdb.tar

  echo "Data copied from 'auth-cache' and 'influxdb'."
}

main() {
  # Main script logic
  echo "Starting backup process for namespace '$namespace'..."
  
  # Check for installed charts in a namespace
  validate_testit_charts_presence "$namespace" 1

  # Variables for backup
  backup_date=$(date +"%d_%m_%Y")
  arch_name="backup.tar"
  pwd_compatible
  backup_path="${work_dir}/backups/${backup_date}"
  mkdir -p "$backup_path" || true

  # Stop deployments
  echo "Stopping TestIT..."
  declare -A replicas
  export replicas
  local testit=("frontend" "webapi" "background-service" "auth" "avatars-api" "ldapwebapi" "license-service" "globalsearch-service")
  stop_deployments_save_replicas "${testit[@]}"

  # Backup services
  echo "Backing up services..."
  backup_services

  # Backup databases
  echo "Backing up databases..."
  backup_postgres

  # Backup minio
  echo "Backing up buckets..."
  backup_minio

  # Start deployments
  echo "Starting TestIT..."
  start_deployments_from_saved_replicas "${testit[@]}"

  echo "Backup process completed successfully!"
  exit 0
}

check_required_variables_array "postgres_vars" "${postgres_vars[@]}"
check_required_variables_array "minio_vars" "${minio_vars[@]}"
command_check "kubectl"
command_check "helm"
command_check "tar"
validate_input "$@" 1
namespace="$1"
validate_namespace "$namespace"

main "$@"
exit 0