#!/bin/bash

# common-functions.sh

# To use functions below in your script, put this file in the same directory and add the line
# source "$(dirname "${BASH_SOURCE[0]}")"/common-functions.sh
# to the top of your script

# Disable debug by default
# ## Usage:
# ## bash ./my-script.sh --debug
debug=false
# Check if the last argument is "--debug"
if [[ "${@: -1}" == "--debug" ]]; then
  debug=true
  set -- "${@:1:$(($#-1))}"  # Remove the last argument from the list
fi
# Enable debugging mode if debug flag is set
if $debug; then
  set -x
fi

# Compatibility for Windows
# ## Usage:
# ## pwd_compatible
pwd_compatible() {
  if [[ "$(expr substr $(uname -s) 1 5)" == "MINGW" ]]; then
    work_dir=$(pwd -W)
  else
    work_dir=$(pwd)
  fi
}

# Function to check if the file exists
# ## Usage:
# ## check_file_existence "/path/to/file"
check_file_existence() {
  local filename="$1"
  if ! [ -f "$filename" ]; then
    echo "The file '$filename' is missing."
    echo "Please check your working directory and make sure the file is present."
    exit 1
  fi
}

# Function to check if a directory exists
# ## Usage:
# ## check_directory_existence "/path/to/dir"
check_directory_existence() {
  local directory_name="$1"
  if ! [ -d "$directory_name" ]; then
    echo "The directory '$directory_name' is missing."
    echo "Please check your working directory and make sure the directory is present."
    exit 1
  fi
}
# Function to check for required dependencies
# ## Usage:
# ## command_check "command"
command_check() {
  local required_command="$1"
  command -v $required_command >/dev/null 2>&1 || {
    echo "'$required_command' command not found. Make sure it's installed and in the PATH."
    exit 1
  }
}

# Function to validate script input
# ## Usage:
# ## validate_input "$@" "2" # - checks if there're 2 inputs provided to the script
validate_input() {
  USAGE_MESSAGE=${USAGE_MESSAGE:-"Usage: $0 <namespace>"}
  local inputs_count="${@: -1}"
  set -- "${@:1:$(($#-1))}"  # Remove the last argument from the list
  if [ $# -ne "$inputs_count" ]; then
    echo $USAGE_MESSAGE
    exit 1
  fi
}

# Function to validate the existence of the specified namespace
# ## Usage:
# ## validate_namespace "my-namespace"
validate_namespace() {
  local namespace="$1"

  # Check if the provided namespace exists
  if ! kubectl get namespace "$namespace" &> /dev/null; then
      echo "Namespace '$namespace' not found."
      exit 1
  fi
}

# Function to check if TestIT charts are present in the provided namespace
# ## Usage:
# ## validate_testit_charts_presence "my-namespace" "2" # - checks if there're 2 TestIT charts present in the namespace
validate_testit_charts_presence(){
  local namespace="$1"
  local charts_count="$2"

  # Check for installed charts in a namespace
  if [[ $(helm list -n $namespace --no-headers | grep unichart | wc -l) -lt "$charts_count" ]]; then
    echo "No TestIT charts detected. Check Helm charts installed in the namespace '$namespace'!"
    helm list -n $namespace
    exit 1
  fi
}

# Function to check if variables are set and not empty
# ## Usage:
# ## postgres_vars=("POSTGRES_HOST" "POSTGRES_PORT" "POSTGRES_USER" "POSTGRES_PASSWORD")
# ## check_required_variables_array "postgres_vars" "${postgres_vars[@]}"
check_required_variables_array() {
  local id="$1"
  shift
  local target_vars=("$@")
  local unset_vars=()
  for var in "${target_vars[@]}"; do
    if [[ -z "${!var:-}" ]]; then
      unset_vars+=("$var")
    fi
  done

  if [[ ${#unset_vars[@]} -gt 0 ]]; then
    echo "Please set the following environment variables in '$id' : ${unset_vars[*]}"
    exit 1
  fi
}

# Function to stop deployments
# ## Usage:
# ## namespace="my-namespace"
# ## deployments=("my-deploy-1" "my-deploy-2")
# ## stop_deployments "${deployments[@]}"
stop_deployments() {
  local s_deployments=("$@")

  for deployment in "${s_deployments[@]}"; do
    kubectl scale deployment "$deployment" -n "$namespace" --replicas 0 > /dev/null 2>&1
  done
  for deployment in "${s_deployments[@]}"; do
    while ! [ -z "$(kubectl get deployment $deployment -n $namespace -o=jsonpath='{.status.readyReplicas}')" ]; do
      sleep 1
    done
  done
}

# Function to start deployments
# ## Usage:
# ## namespace="my-namespace"
# ## deployments=("my-deploy-1" "my-deploy-2")
# ## start_deployments "${deployments[@]}"
start_deployments() {
  local s_deployments=("$@")
  for deployment in "${s_deployments[@]}"; do
    kubectl scale deployment "$deployment" -n "$namespace" --replicas 1 > /dev/null 2>&1
  done

  for deployment in "${s_deployments[@]}"; do
    echo "Waiting for deployment '$deployment'..."
    # Set counter
    local counter=0
    local timeout=180
    # Wait for pods till timeout
    while [[ "$(kubectl get deployment "$deployment" -n "$namespace" -o=jsonpath='{.status.readyReplicas}')" != "1" ]]; do
      sleep 1
      ((counter+=1))
      if [[ "$counter" == "$timeout" ]]; then
        echo "Deployment '$deployment' is taking longer than ${timeout}s to become ready."
        break
      fi
    done
  done
}

# Function to stop deployments saving replicaCount
# ## !!!! replicas array should be defined in the main script like below:
# ## Usage:
# ## namespace="my-namespace"
# ## deployments=("my-deploy-1" "my-deploy-2")
# ## declare -A replicas && export replicas
# ## stop_deployments_save_replicas "${deployments[@]}"
stop_deployments_save_replicas() {
  local s_deployments=("$@")
  for deployment in "${s_deployments[@]}"; do
    # Save current replicaCount
    replicas["$deployment"]=$(kubectl get deployment "$deployment" -n "$namespace" -o=jsonpath='{.spec.replicas}')
    # Stop all pods
    kubectl -n "$namespace" scale deployment "$deployment" --replicas 0 1>/dev/null
  done
  for deployment in "${s_deployments[@]}"; do
    # Wait till all pods are down
      echo "Waiting for pods termination of deployment '$deployment'..."
    while [ "$(kubectl get pods -l app=$deployment -n $namespace --no-headers 2>/dev/null | wc -l)" -ne 0 ]; do
      sleep 5
    done
  done
}

# Function to start deployments back up to saved replicaCount
# ## !!!! WORKS IN PAIR WITH stop_deployments_save_replicas
# ## Usage:
# ## namespace="my-namespace"
# ## deployments=("my-deploy-1" "my-deploy-2")
# ## start_deployments_from_saved_replicas "${deployments[@]}"
start_deployments_from_saved_replicas() {
  local s_deployments=("$@")
  # Restore ReplicaCount
  for deployment in "${s_deployments[@]}"; do
    # Skip current deployment if saved replicaCount was 0
    if [[ "${replicas["$deployment"]}" -eq 0 ]]; then continue; fi
    # Restore replicaCount
    kubectl scale deployment "$deployment" -n "$namespace" --replicas "${replicas["$deployment"]}" 1>/dev/null
  done
  # Wait for the desired amount of replicas
  for deployment in "${s_deployments[@]}"; do
    # Skip current deployment if saved replicaCount was 0
    if [[ "${replicas["$deployment"]}" -eq 0 ]]; then continue; fi
    echo "Waiting for deployment '$deployment'..."
    # Set counter
    local counter=0
    local timeout=180
    # Wait for pods till timeout
    while [[ "$(kubectl get deployment "$deployment" -n "$namespace" -o=jsonpath='{.status.readyReplicas}')" != "${replicas["$deployment"]}" ]]; do
      sleep 1
      ((counter+=1))
      if [[ "$counter" == "$timeout" ]]; then
        echo "Deployment '$deployment' is taking longer than ${timeout}s to become ready."
        break
      fi
    done
  done
}

# Function to update deployments image
# ## Usage:
# ## namespace="my-namespace"
# ## deployments=("my-deploy-1" "my-deploy-2")
# ## update_deployments_image "$new_image" "${deployments[@]}"
update_deployments_image() {
  local image="$1"
  shift
  local u_deployments=("$@")
  for deployment in "${u_deployments[@]}"; do
    echo "Updating deployment ${deployment} with image ${image}..."
    kubectl scale deployment "$deployment" -n "$namespace"  --replicas 1 > /dev/null 2>&1
    local container=$(kubectl get deployment $deployment -n $namespace -o=jsonpath='{.spec.template.spec.containers[0].name}')
    kubectl set image deployment/$deployment $container=$image -n $namespace
  done

  for deployment in "${u_deployments[@]}"; do
    echo "Waiting for deployment '$deployment'..."
    local counter=0
    local timeout=180
    while [[ -z "$(kubectl get deployment "$deployment" -n "$namespace" -o=jsonpath='{.status.readyReplicas}')" ]]; do
      sleep 1
      ((counter+=1))
      if [[ "$counter" == "$timeout" ]]; then
        echo "Deployment '$deployment' is taking longer than ${timeout}s to become ready."
        break
      fi
    done
  done
}

# Function to get pod name by label
# ## Usage:
# ## namespace="my-namespace"
# ## pod_by_label "app=postgres"
pod_by_label() {
  local label="$1"
  pod_name=$(kubectl get pods -l "$label" -n "$namespace"  --output=jsonpath='{.items[0].metadata.name}')
  echo $pod_name
}

# Function to get an environment variable from a Kubernetes pod
# ## Usage:
# ## namespace="my-namespace"
# ## get_env_variable_from_pod "postgres" "POSTGRES_USER"
get_env_variable_from_pod() {
  local deployment="$1"
  local variable_name="$2"
  local pod_name=$(pod_by_label "app=$deployment")
  if [ -z "$pod_name" ]; then
    exit 1
  fi
  # Use kubectl exec to run the printenv command inside the pod and fetch the environment variable
  local env_value=$(kubectl exec -n "$namespace" "$pod_name" -- printenv "$variable_name" 2>/dev/null)

  # Output the value of the environment variable
  echo "$env_value"
}

# Function to copy data to pod
# ## Usage:
# ## namespace="my-namespace"
# ## copy_to_pod "pod_name" "local_file" "destination_file_path"
copy_to_pod() {
  local pod="$1"
  local file="$2"
  local dest="$3"
 
  kubectl cp -n "$namespace" "${file}" "${pod}:${dest}" --retries 999 --no-preserve > /dev/null 2>&1
}

# Function to execute psql command in pod
# ## Usage:
# ## namespace="my-namespace"
# ## execute_psql "$pod_name" "$psql_user" "$database" "$psql_query"
execute_psql() {
  local pod_name="$1"
  local user="$2"
  local db="$3"
  local query="$4"

  echo "Executing query '$query' in pod '$pod_name'..."
  kubectl exec "$pod_name" -n "$namespace" --  psql -U "$user" -d "$db" -c "$query"
}
